<?php

namespace App\Calculators;

use App\Models\Option;
use App\Models\Question;
use Illuminate\Support\Facades\Log;

class FormulaCalculator
{
    /**
     * ✅ متد اصلی و عمومی برای محاسبه هر نوع فرمول (برای استفاده در کنترلرها و تست‌ها)
     */
    public function calculate(string $expression, array $variables = [], array $formulaResults = []): array
    {
        try {
            // ۱. جایگزینی متغیرهای سوال: {Q1} → 10
            $normalized = $this->normalizeExpression($expression, $variables);

            // ۲. جایگزینی نتایج فرمول‌های دیگر: {MATH_ADD} → 15
            $normalized = $this->normalizeFormulaReferences($normalized, $formulaResults);

            // ۳. تبدیل عملگرهای خاص به معادل PHP
            $normalized = $this->transformExpression($normalized);

            // ۴. ایمن‌سازی
            if (!$this->isSafeExpression($normalized)) {
                throw new \Exception('Expression contains unsafe characters');
            }

            // ۵. اجرای نهایی
            $result = $this->evaluate($normalized);

            return [
                'success' => true,
                'value' => $result,
                'type' => gettype($result),
            ];

        } catch (\Throwable $e) {
            return [
                'success' => false,
                'error' => $e->getMessage(),
                'value' => null,
            ];
        }
    }

    /**
     * جایگزینی ارجاع به فرمول‌های دیگر
     * مثال: {MATH_ADD} → 15
     */
    protected function normalizeFormulaReferences(string $expression, array $formulaResults): string
    {
        // اگر آرایه خالی است، کاری نکن
        if (empty($formulaResults)) {
            return $expression;
        }

        foreach ($formulaResults as $code => $result) {
            // فقط اگر کد معتبر است و نتیجه موفق بوده
            if (!empty($code) && ($result['success'] ?? false)) {
                $value = $result['value'];

                // اگر مقدار رشته است، آن را در نقل‌قول بگذار
                if (is_string($value)) {
                    $value = '"' . addslashes($value) . '"';
                }

                // جایگزینی دقیق {CODE} با مقدار
                // استفاده از preg_replace برای جلوگیری از تداخل با متغیرهای دیگر
                $pattern = '/\{' . preg_quote($code, '/') . '\}/';
                $expression = preg_replace($pattern, $value, $expression);
            }
        }

        return $expression;
    }
    /**
     * ✅ متد اختصاصی برای محاسبه فرمول سوال (برای استفاده در لاجیک قدیمی پروژه)
     */
    public function calculateQuestionFormula(Question $question, array $answers): float
    {
        $formula = $question->formula;
        if (!$formula) return 0;

        $expression = $formula->formula_expression;
        $config = $formula->formula_config ?? [];

        // ۱. جایگزینی متغیرها با مقادیر واقعی
        foreach ($config as $varName => $questionId) {
            $score = $this->getQuestionScore($questionId, $answers);
            $expression = str_replace('{' . $varName . '}', (string)$score, $expression);
        }

        // ۲. تبدیل عملگرهای خاص به معادل PHP
        $expression = $this->transformExpression($expression);

        // ۳. ایمن‌سازی
        if (!$this->isSafeExpression($expression)) {
            Log::error('Unsafe expression detected', ['expression' => $expression]);
            return 0;
        }

        // ۴. اجرا
        try {
            $result = eval('return ' . $expression . ';');
            return is_numeric($result) ? (float)$result : 0;
        } catch (\Throwable $e) {
            Log::error('Formula calculation failed', [
                'expression' => $expression,
                'error' => $e->getMessage(),
                'question_id' => $question->id,
            ]);
            return 0;
        }
    }

    /**
     * گرفتن نمره یک سوال از آرایه answers
     */
    protected function getQuestionScore(int $questionId, array $answers): float
    {
        $answer = $answers[$questionId] ?? null;
        if ($answer && !empty($answer['option_id'])) {
            $option = Option::find($answer['option_id']);
            return $option ? (float)$option->score_value : 0;
        }
        return 0;
    }

    /**
     * تبدیل {Q1} به مقدار واقعی
     */
    protected function normalizeExpression(string $expression, array $variables): string
    {
        foreach ($variables as $key => $value) {
            $search = '{' . $key . '}';
            if (is_string($value)) {
                $escaped = addslashes($value);
                $expression = str_replace($search, "\"{$escaped}\"", $expression);
            } else {
                $expression = str_replace($search, $value, $expression);
            }
        }
        return $expression;
    }

    /**
     * تبدیل عملگرهای غیرذاتی به معادل PHP
     */
    protected function transformExpression(string $expression): string
    {
        // تبدیل in به in_array
        $expression = preg_replace(
            '/(\d+\.?\d*|\w+)\s+in\s+\[([^\]]+)\]/',
            'in_array($1, [$2])',
            $expression
        );

        // تبدیل contains به str_contains
        $expression = preg_replace(
            '/contains\s*\(\s*([^,]+)\s*,\s*([^)]+)\s*\)/',
            'str_contains($1, $2)',
            $expression
        );

        // تبدیل توابع رشته‌ای
        $expression = preg_replace('/\blen\s*\(/', 'strlen(', $expression);
        $expression = preg_replace('/\bupper\s*\(/', 'strtoupper(', $expression);
        $expression = preg_replace('/\blower\s*\(/', 'strtolower(', $expression);

        // تبدیل توابع آماری
        $expression = preg_replace('/\bavg\s*\(([^)]+)\)/', 'array_sum([$1]) / count([$1])', $expression);
        $expression = preg_replace('/\bsum\s*\(([^)]+)\)/', 'array_sum([$1])', $expression);
        $expression = preg_replace('/\bmin\s*\(([^)]+)\)/', 'min([$1])', $expression);
        $expression = preg_replace('/\bmax\s*\(([^)]+)\)/', 'max([$1])', $expression);

        // تبدیل between
        $expression = preg_replace(
            '/(\w+)\s+between\s+(\d+)\s+and\s+(\d+)/',
            '$1 >= $2 && $1 <= $3',
            $expression
        );

        return $expression;
    }

    /**
     * بررسی امنیت عبارت
     */
    protected function isSafeExpression(string $expression): bool
    {
        // ✅ الگوی جدید با پشتیبانی از فارسی و یونیکد
        $pattern = '/^[\d\s\+\-\*\/\%\^\(\)\>\<\=\!\&\|\?:\',\"\.a-zA-Z_\[\]{}\x{0600}-\x{06FF}]+$/u';

        // ✅ لیست توابع مجاز (اضافه کردن pow و سایر توابع ریاضی)
        $allowedFunctions = [
            // توابع آرایه‌ای
            'in_array', 'array_sum', 'count', 'min', 'max',

            // توابع رشته‌ای
            'str_contains', 'strlen', 'strtoupper', 'strtolower',

            // ✅ توابع ریاضی (جدید)
            'pow', 'abs', 'round', 'floor', 'ceil', 'sqrt', 'log', 'exp',

            // توابع شرطی
            'if',
        ];

        // چک کردن وجود توابع غیرمجاز
        if (preg_match_all('/(\w+)\s*\(/', $expression, $matches)) {
            foreach ($matches[1] as $func) {
                if (!in_array(strtolower($func), $allowedFunctions)) {
                    return false; // ❌ تابع غیرمجاز
                }
            }
        }

        return preg_match($pattern, $expression) === 1;
    }
    /**
     * اجرای نهایی با eval
     */
    protected function evaluate(string $expression)
    {
        $result = null;
        $error = null;

        set_error_handler(function($errno, $errstr) use (&$error) {
            $error = $errstr;
            return true;
        });

        try {
            $code = "return ($expression);";
            $result = eval($code);
        } catch (\Throwable $e) {
            $error = $e->getMessage();
        }

        restore_error_handler();

        if ($error) {
            throw new \Exception("Evaluation error: {$error}");
        }

        return $result;
    }

    /**
     * تست سریع برای دیباگ
     */
    public function test(string $expression, array $variables = []): void
    {
        $result = $this->calculate($expression, $variables);
        echo "Expression: {$expression}\n";
        echo "Variables: " . json_encode($variables, JSON_UNESCAPED_UNICODE) . "\n";
        echo "Result: " . print_r($result, true) . "\n";
        echo str_repeat('-', 50) . "\n";
    }

    public function show($testId, $attemptId)
    {
        \Log::info('=== show() called ===', [
            'testId' => $testId,
            'attemptId' => $attemptId,
            'view_path' => resource_path('views/admin/formula-results/show.blade.php'),
        ]);

        $test = Test::findOrFail($testId);
        $results = FormulaResult::where('test_id', $testId)
            ->where('attempt_id', $attemptId)
            ->orderBy('created_at')
            ->get();

        \Log::info('=== Returning view ===', [
            'results_count' => $results->count(),
        ]);

        return view('admin.formula-results.show', compact('test', 'attemptId', 'results'));
    }
}
