<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use App\Models\Formula;
use App\Models\Test;
use App\Calculators\FormulaCalculator;

class FormulaTestCommand extends Command
{
    protected $signature = 'formula:test {testId}';
    protected $description = 'تست تمام فرمول‌های یک تست و تأیید صحت محاسبات';

    public function handle()
    {
        $testId = $this->argument('testId');
        $test = Test::find($testId);

        if (!$test) {
            $this->error("❌ تستی با ID {$testId} یافت نشد!");
            return Command::FAILURE;
        }

        $this->info("\n🧪 شروع تست فرمول‌های: {$test->title}\n");

        $calculator = new FormulaCalculator();
        $formulas = Formula::where('test_id', $testId)->get();

        if ($formulas->isEmpty()) {
            $this->warn("⚠️ هیچ فرمولی برای این تست یافت نشد!");
            $this->info("ابتدا از دستور زیر استفاده کن:");
            $this->info("   php artisan db:seed --class=FormulaTestSeeder\n");
            return Command::FAILURE;
        }

        $this->table(
            ['وضعیت', 'کد', 'نام', 'فرمول', 'انتظار', 'نتیجه', 'توضیحات'],
            $this->runTests($formulas, $calculator)
        );

        return Command::SUCCESS;
    }

    protected function runTests($formulas, $calculator)
    {
        $results = [];

        // مقادیر مورد انتظار برای هر فرمول (از Seeder)
        $expectedValues = $this->getExpectedValues();

        foreach ($formulas as $formula) {
            $config = is_string($formula->formula_config)
                ? json_decode($formula->formula_config, true)
                : ($formula->formula_config ?? []);

            $result = $calculator->calculate($formula->formula_expression, $config);

            $expected = $expectedValues[$formula->code] ?? null;
            $actual = $result['value'] ?? null;
            $success = $result['success'] ?? false;

            // مقایسه نتیجه با مقدار مورد انتظار
            $isCorrect = $success && ($actual == $expected);

            $results[] = [
                $isCorrect ? '✅' : '❌',
                $formula->code,
                $formula->name,
                $formula->formula_expression,
                $expected,
                $actual,
                $result['error'] ?? '-'
            ];
        }

        return $results;
    }

    protected function getExpectedValues()
    {
        // این مقادیر باید با FormulaTestSeeder هماهنگ باشند
        return [
            'MATH_ADD' => 3,
            'MATH_SUB' => 6,
            'MATH_MUL' => 15,
            'MATH_DIV' => 5,
            'MATH_MOD' => 1,
            'MATH_POW' => 25,
            'CMP_GT' => 1,
            'CMP_LT' => 1,
            'CMP_EQ' => 1,
            'CMP_NEQ' => 1,
            'CMP_GTE' => 1,
            'CMP_LTE' => 1,
            'LOG_AND' => 1,
            'LOG_OR' => 1,
            'LOG_NOT' => 1,
            'TERN_SIMPLE' => 20,
            'TERN_FALSE' => 3,
            'TERN_NESTED' => 'Medium',
            'MEM_IN_TRUE' => 1,
            'MEM_IN_FALSE' => 0,
            'STR_CONTAINS_TRUE' => 1,
            'STR_CONTAINS_FALSE' => 0,
            'STR_LEN' => 1,
            'STAT_AVG' => 20,
            'STAT_SUM' => 60,
            'STAT_MIN' => 10,
            'STAT_MAX' => 30,
            'BETWEEN_TRUE' => 1,
            'BETWEEN_FALSE' => 0,
            'COMBO_1' => 26,
            'COMBO_2' => 'High',
        ];
    }
}
