<?php

namespace App\Http\Controllers;

use App\Calculators\FormulaCalculator;
use App\Models\Test;
use App\Models\TestAttempt;
use App\Models\Answer;
use App\Models\Option;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class TestController extends Controller
{
    /**
     * نمایش فرم تست (GET)
     */
    public function show(Test $test)
    {
        $test->load(['questions.options' => function ($query) {
            $query->orderBy('id');
        }]);

        return view('test.form', compact('test'));
    }

    /**
     * ثبت پاسخ‌ها (POST)
     */
    public function submit(Request $request, Test $test, FormulaCalculator $calculator)
{
    $validated = $request->validate([
        'participant_name' => 'required|string|max:255',
        'participant_phone' => 'required|string|max:20',
        'answers.*.question_id' => 'required|integer|exists:questions,id',
        'answers.*.option_id' => 'nullable|integer|exists:options,id',
        'answers.*.text_answer' => 'nullable|string|max:1000',
    ]);

    return DB::transaction(function () use ($request, $test, $validated, $calculator) {
        
        // ✅ محاسبه فرمول هر سوال جداگانه
        $questionResults = $calculator->calculateAllQuestions($test->id, $validated['answers']);

        // نمره کل = جمع نتایج همه سوالات
        $totalScore = array_sum(array_column($questionResults, 'score'));

        $attempt = TestAttempt::create([
            'user_id' => 1,
            'test_id' => $test->id,
            'participant_name' => $validated['participant_name'],
            'participant_phone' => $validated['participant_phone'],
            'status' => 'completed',
            'total_score' => $totalScore,
            'completed_at' => now(),
        ]);

        $answersData = [];
        foreach ($validated['answers'] as $questionId => $answer) {
            $answersData[] = [
                'attempt_id' => $attempt->id,
                'question_id' => $questionId,
                'option_id' => $answer['option_id'] ?? null,
                'text_answer' => $answer['text_answer'] ?? null,
                'created_at' => now(),
                'updated_at' => now(),
            ];
        }

        foreach ($answersData as &$answer) {
            $answer['attempt_id'] = $attempt->id;
        }

        Answer::upsert($answersData, ['attempt_id', 'question_id'], ['option_id', 'text_answer', 'updated_at']);

        return redirect()->route('tests.show', $test->id)->with([
            'success' => 'تست با موفقیت ثبت شد',
            'total_score' => $totalScore,
            'question_results' => $questionResults,  // ✅ نتیجه هر سوال
        ]);
    });
}
}